#!/usr/bin/env python
import pygame
import random
import os.path

DOWN = 0
LEFT = 1
RIGHT = 2
UP = 3

TILE = 16
SCALE = 3
SCALED_TILE = TILE * SCALE

def dirToVel(direction, speed, offset = 0.0):
	if direction == DOWN:
		return (offset, speed)
	elif direction == LEFT:
		return (-speed, offset)
	elif direction == RIGHT:
		return (speed, offset)
	elif direction == UP:
		return (offset, -speed)


class SoundManager():
	def __init__(self):
		self.fire = pygame.mixer.Sound(os.path.join('data', 'sounds', 'fire.wav'))
		self.fire.set_volume(0.5)
		self.splat = pygame.mixer.Sound(os.path.join('data', 'sounds', 'splat.wav'))
		self.clone = pygame.mixer.Sound(os.path.join('data', 'sounds', 'clone.wav'))
		self.clone.set_volume(0.8)
		self.death = pygame.mixer.Sound(os.path.join('data', 'sounds', 'death.wav'))



class Wall(pygame.sprite.Sprite):
	def __init__(self, (x, y)):
		pygame.sprite.Sprite.__init__(self)
		self.image = pygame.transform.scale(pygame.image.load(os.path.join('data', 'wall.bmp')).convert(), (SCALED_TILE, SCALED_TILE))
		self.pos = (x, y)
		self.rect = pygame.Rect(self.pos, self.image.get_size())
	
	def update(self, screen):
		screen.blit(self.image, self.pos)

	def get_rect(self):
		return pygame.Rect(self.pos, self.image.get_size())
		

class Player(pygame.sprite.Sprite):
	def __init__(self, (x, y)):
		pygame.sprite.Sprite.__init__(self)
		self.image = pygame.transform.scale(pygame.image.load(os.path.join('data', 'player.bmp')).convert(), (SCALED_TILE * 4, SCALED_TILE * 4))
		self.image.set_colorkey((0, 128, 128))
		self.rect = pygame.Rect((x, y), (SCALED_TILE / 2, SCALED_TILE / 2))
		self.vel = [0, 0]
		self.anim_timer = 0
		self.frame = 0
		self.is_moving = False
		self.direction = DOWN
		self.attack = False
		self.cooldown = 0
		self.attack_off = 0
		self.attack_vel = 1

	def handle_events(self):
		speed = 4
		pressed = pygame.key.get_pressed()
		clicked = pygame.mouse.get_pressed()
		d = -1 

		if pressed[pygame.K_SPACE] or clicked[0]:
			self.attack = True
			speed = 3

		if pressed[pygame.K_UP]:
			d = UP
		elif pressed[pygame.K_DOWN]:
			d = DOWN
		elif pressed[pygame.K_LEFT]:
			d = LEFT
		elif pressed[pygame.K_RIGHT]:
			d = RIGHT

		if d is not -1:
			self.direction = d
			self.vel = dirToVel(self.direction, speed)
	
	def update(self, screen, objects, sm):
		self.handle_events()
		# Move
		if self.vel != [0, 0]:
			self.rect.x += self.vel[0]
			self.rect.y += self.vel[1]

			if not screen.get_rect().contains(self.rect) or pygame.sprite.spritecollide(self, objects['wall'], False):
				self.rect.x -= self.vel[0]
				self.rect.y -= self.vel[1]
			else:
				self.is_moving = True

			self.vel = [0, 0]

		if pygame.sprite.spritecollide(self, objects['enemy'], True):
			for i in range(8):
				pos = (self.rect.x - 4 + random.randint(-32, 32), self.rect.y - 8 + random.randint(-32, 32))
				objects['effect'].add(Splat(pos, True))
			sm.death.play()
			self.kill()


		# Attack
		if self.attack and self.cooldown <= 0:
			pos = (self.rect.center[0] + dirToVel(self.direction, 8 * SCALE)[0], self.rect.center[1] + dirToVel(self.direction, 8 * SCALE)[1])
			if self.direction is not UP and self.direction is not DOWN:
				pos = (pos[0], pos[1] - 8)
			elif self.direction is UP:
				pos = (pos[0] - 16, pos[1])
			if len(objects['player']) and objects['player'].sprites()[0] == self:
				sm.fire.play()
			objects['bullet'].add(Bullet(pos, dirToVel(self.direction, 5, self.attack_off)))
			self.attack_off += self.attack_vel
			if abs(self.attack_off):
				self.attack_vel *= -1
			self.cooldown = 10

		self.attack = False
		if self.cooldown > 0:
			self.cooldown -= 1

		# Anim
		if self.is_moving:
			self.anim_timer += 1
			if self.anim_timer > 6:
				self.frame += 1
				if self.frame > 3: self.frame = 0
				self.anim_timer = 0
		else:
			self.frame = 0
			self.anim_timer = 0

		# Blit
		rect = pygame.Rect(0 + SCALED_TILE * self.frame, SCALED_TILE * self.direction, SCALED_TILE, SCALED_TILE)
		pos = (self.rect.x - 4 * SCALE, self.rect.y - 8 * SCALE)
		screen.blit(self.image, pos, rect)
		self.is_moving = False


class Spawner(pygame.sprite.Sprite):
	def __init__(self, (x, y)):
		pygame.sprite.Sprite.__init__(self)
		self.image = pygame.transform.scale(pygame.image.load(os.path.join('data', 'spawner.bmp')), (SCALED_TILE, SCALED_TILE))
		self.image.set_colorkey((255, 0, 255))
		self.rect = pygame.Rect((x, y), self.image.get_size())
		self.timer = random.randint(60, 180)
		self.wave = 0
		self.wave_timer = 0
		self.tone = 0
		self.bonus_count = random.randint(2, 10)
	
	def update(self, screen, objects):
		self.wave_timer += 1
		if self.wave_timer > 60 * 30:
			if self.wave < 3:
				self.wave += 1
			self.wave_timer = 0

		self.timer -= 1
		if self.timer <= 0:
			bonus = False
			self.bonus_count -= 1
			if self.bonus_count <= 0:
				bonus = True
				self.bonus_count = 10 + 5 * self.wave

			objects['enemy'].add(Enemy((self.rect.topleft), bonus))

			if self.wave_timer >= 60 * 20:
				if self.wave == 3:
					self.timer = 30
				else:
					self.timer = random.randint(60 - 20 * self.wave, 180 - 40 * self.wave)
			else:
				self.timer = random.randint(120, 360)

		self.tone += 8
		if self.tone > 255:
			self.tone -= 255
		self.image.set_palette_at(0, (abs(self.tone), 0, 0))
		screen.blit(self.image, self.rect.topleft)
			
		

class Enemy(pygame.sprite.Sprite):
	def __init__(self, (x, y), bonus):
		pygame.sprite.Sprite.__init__(self)
		self.image = pygame.transform.scale(pygame.image.load(os.path.join('data', 'enemy.bmp')), (SCALED_TILE / 2, SCALED_TILE / 2))
		self.image.set_colorkey((255, 0, 255))
		self.rect = pygame.Rect((x, y), self.image.get_size())
		self.direction = random.randint(0, 3)
		self.timer = random.randint(30, 180)
		self.angle = random.randint(0, 360)
		self.bonus = bonus
	
	def update(self, screen, objects, game_info, sm):
		self.timer -= 1
		if not self.timer:
			self.direction = random.randint(0, 3)
			self.timer = random.randint(30, 180)
		self.rect.move_ip(dirToVel(self.direction, 3))

		if pygame.sprite.spritecollide(self, objects['wall'], False) or not screen.get_rect().contains(self.rect):
			self.rect.move_ip(dirToVel(self.direction, -3))
			self.direction = random.randint(0, 3)

		if pygame.sprite.spritecollide(self, objects['bullet'], True):
			game_info['score'] += 100
			if self.bonus:
				objects['player'].add(Player(self.rect.topleft))
				sm.clone.play()
			sm.splat.play()
			objects['effect'].add(Splat((self.rect.x - 4, self.rect.y - 4), False))
			self.kill()

		if self.bonus:
			for player in objects['player']:
				pygame.draw.line(screen, (192, 0, 0), self.rect.center, (player.rect.center[0], player.rect.center[1] - 4), 2)

		self.angle += 8
		if self.angle >= 360:
			self.angle -= 360
		rotated_image = pygame.transform.rotate(self.image, self.angle)
		screen.blit(rotated_image, self.rect.topleft)


class Bullet(pygame.sprite.Sprite):
	def __init__(self, (x, y), (vel_x, vel_y)):
		pygame.sprite.Sprite.__init__(self)
		image = pygame.image.load(os.path.join('data', 'bullet.bmp')).convert()
		self.image = pygame.transform.scale(image, (image.get_width() * SCALE, image.get_height() * SCALE))
		self.image.set_colorkey((255, 0, 255))
		self.rect = pygame.Rect((x, y), self.image.get_size())
		self.vel = (vel_x, vel_y)
		self.timer = 120
	
	def update(self, screen, objects):
		self.rect.move_ip(self.vel)
		self.timer -= 1
		pos = (self.rect.x - 2 * SCALE, self.rect.y - 2 * SCALE)
		if not self.timer or pygame.sprite.spritecollide(self, objects['wall'], False):
			objects['effect'].add(BulletParticle(pos))
			self.kill()
		screen.blit(self.image, self.rect.topleft)


class BulletParticle(pygame.sprite.Sprite):
	def __init__(self, (x, y)):
		pygame.sprite.Sprite.__init__(self)
		self.image = pygame.transform.scale(pygame.image.load(os.path.join('data', 'bullet_particle.bmp')).convert(), (SCALED_TILE / 2 * 4, SCALED_TILE / 2))
		self.image.set_colorkey((255, 0, 255))
		self.rect = pygame.Rect((x, y), self.image.get_size())
		self.anim_timer = 0
		self.frame = 0
	
	def update(self, screen):
		self.anim_timer += 1
		if self.anim_timer > 2:
			self.anim_timer = 0
			self.frame += 1
			if self.frame > 3:
				self.kill()

		rect = pygame.Rect((self.frame * SCALED_TILE / 2, 0), (SCALED_TILE / 2, SCALED_TILE / 2))
		screen.blit(self.image, self.rect, rect)


class Splat(pygame.sprite.Sprite):
	def __init__(self, (x, y), player):
		pygame.sprite.Sprite.__init__(self)
		if player:
			image = pygame.image.load(os.path.join('data', 'player_splat.bmp'))
		else:
			image = pygame.image.load(os.path.join('data', 'enemy_splat.bmp'))
		image.set_colorkey((255, 0, 255))
		self.image = pygame.transform.scale(image.convert(), (SCALED_TILE * 4, SCALED_TILE))
		self.rect = pygame.Rect((x, y), self.image.get_size())
		self.anim_timer = 0
		self.frame = 0
	
	def update(self, screen):
		self.anim_timer += 1
		if self.anim_timer > 2:
			self.anim_timer = 0
			self.frame += 1
			if self.frame > 3:
				self.kill()

		rect = pygame.Rect((self.frame * SCALED_TILE, 0), (SCALED_TILE, SCALED_TILE))
		screen.blit(self.image, self.rect, rect)


def load_level(name, objects):
	level = pygame.image.load(name)
	for y in range(level.get_height()):
		for x in range(level.get_width()):
			c = level.get_at((x, y))
			if c == (0, 0, 0):
				objects['wall'].add(Wall((x * SCALED_TILE, y * SCALED_TILE)))
			elif c == (0, 0, 255):
				objects['player'].add(Player((x * SCALED_TILE, y * SCALED_TILE)))
			elif c == (255, 0, 0):
				objects['spawner'].add(Spawner((x * SCALED_TILE, y * SCALED_TILE)))


def reload_game(objects, game_info):
	for group in objects.itervalues():
		group.empty()
	
	game_info['score'] = 0
	load_level('levels/default.bmp', objects)
	

def main():
	pygame.mixer.pre_init(44100, -16, 8, 1024)
	pygame.init()
	size = (256 * SCALE, 256 * SCALE)

	screen = pygame.display.set_mode(size)
	pygame.mouse.set_visible(False)

	font = pygame.font.Font(None, 32)

	ground_image = pygame.transform.scale(pygame.image.load(os.path.join('data', 'ground.bmp')).convert(), (SCALED_TILE, SCALED_TILE))
	ground = pygame.Surface(size)
	for i in range(16 * 16):
		ground.blit(ground_image, (i / 16 * SCALED_TILE, i % 16 * SCALED_TILE))

	wall_group = pygame.sprite.Group()
	enemy_group = pygame.sprite.Group()
	spawner_group = pygame.sprite.Group()
	bullet_group = pygame.sprite.Group()
	player_group = pygame.sprite.Group()
	effect_group = pygame.sprite.Group()
	objects = {'wall': wall_group, 'enemy': enemy_group, 'player': player_group, 'bullet': bullet_group, 'effect': effect_group, 'spawner': spawner_group}

	game_info = {'score': 0, 'highscore': 0}

	sm = SoundManager()

	load_level('levels/default.bmp', objects)
	filter_image = pygame.transform.scale(pygame.image.load(os.path.join('data', 'filter.png')).convert_alpha(), size)

	run = True
	game_over = False
	clock = pygame.time.Clock()

	capture = False
	shot = 0

	while run:
		clock.tick(60)
		caption = "Virus Hunter [FPS:%.2f]" % clock.get_fps()
		pygame.display.set_caption(caption)

		if not player_group:
			game_over = True

		for event in pygame.event.get():
			if event.type == pygame.QUIT: run = False
			if event.type == pygame.KEYDOWN and event.key == pygame.K_ESCAPE: run = False
			if event.type == pygame.KEYDOWN and event.key == pygame.K_c: capture = not(capture)
			if game_over and event.type == pygame.KEYDOWN and event.key == pygame.K_RETURN:
				reload_game(objects, game_info)
				game_over = False

		screen.blit(ground, (0, 0))
		objects['wall'].update(screen)
		objects['spawner'].update(screen, objects)
		objects['bullet'].update(screen, objects)
		objects['enemy'].update(screen, objects, game_info, sm)
		objects['player'].update(screen, objects, sm)
		objects['effect'].update(screen)
		screen.blit(filter_image, (0, 0))

		text = font.render("Score: %i" % game_info['score'], 0, (255, 255, 255)).convert()
		text_scaled = pygame.transform.scale(text, (text.get_width() * SCALE / 2, text.get_height() * SCALE / 2))
		pos = (10, screen.get_height() - text_scaled.get_height() - 10)
		screen.blit(text_scaled, pos)

		if not player_group:
			text = font.render("Game Over", 0, (255, 255, 255)).convert()
			text_scaled = pygame.transform.scale(text, (text.get_width() * SCALE, text.get_height() * SCALE))
			pos = (screen.get_width() / 2 - text_scaled.get_width() / 2, screen.get_height() / 2 - text_scaled.get_height() / 2)
			screen.blit(text_scaled, pos)

			text = font.render("Press Enter to continue", 0, (255, 255, 255)).convert()
			text_scaled = pygame.transform.scale(text, (text.get_width() * SCALE / 2, text.get_height() * SCALE / 2))
			pos = (screen.get_width() / 2 - text_scaled.get_width() / 2, screen.get_height() / 2 - text_scaled.get_height() / 2 + 60)
			screen.blit(text_scaled, pos)

		pygame.display.update()

		if capture:
			shot += 1
			pygame.image.save(screen, "gif/capture%i.png" % shot)

	pygame.quit()

if __name__ == "__main__":
	main()
